#!/usr/bin/perl

use strict;
use warnings;

use Getopt::Long;
use WWW::Telegram::BotAPI;
use File::Spec::Functions;
use Cwd 'abs_path';
use Time::Local;
use File::Path qw(make_path remove_tree);
use Time::TZOffset qw/tzoffset/;
use File::Copy;
use Encode qw 'from_to encode decode';

my $vers = '0.0.0.1';

abs_path($0) =~ /^(.*?)[\\\/]([^\\\/]+)$/;
my ( $curpath, $programfile ) = ( $1, $2 );

my ( $logfile, $configfile, %config, $needhelp, $warnnodelist, $warnnetmail,
	$newmailcount, $daysold, @tgmsgs );


sub usage()
{
    print <<US;
    
    $programfile warns SysOp about new mail or if nodlist is old.
                                          Written by Stas Mishchenkov 2:460/58.

    Usage: $programfile [options]
    Options:
        -h,--help               this text
        -l,--log[=]filename     log file name. Optional.
        -n,--nodelist           warn if nodelist is old.
        -m,--mail               warn if new netmail.
        -c,--config[=]filename  configuration file name. Required.
US
exit;
}

sub readconf()
{
    my ( $F, $line, $key );
    return 0 unless defined $configfile;
    print "Reading config file $configfile\n";
    unless( open( $F, '<', $configfile ) ) {
		print STDERR "Can't open config file $configfile ($!).";
		exit;
    }
    my $lines = 0;
    while( $line = <$F> ) {
		$lines++;
		$line =~ s/[\r\n]//g;
		$line =~ s/	/ /g;
		next if $line =~ /^\s*[\#\;]/;
		next if $line =~ /^\s+$/;
		next if $line eq '';
		next unless $line =~ /\s*(\S+)\s+(.*?)\s*$/;
		$key = lc($1);
		$config{$key} = $2;
    }
    close($F);
    print "$lines lines read.\n";

	if ( defined $config{fromaddress} ) {
		if ( $config{fromaddress} =~ /^(\d+)\:\d+\/\d+\.?\d*/ ) {
			$config{defaultzone} = $1;
		} else {
			writelog( 'FromAddress must be fidonet address.' );
			exit;
		}
	} else {
		$config{defaultzone} = 2;
	}
	unless ( defined $config{nodelist} ) {
		writelog( 'Nodelist must be defined!' );
		exit;
	}
	unless ( defined( $config{pktpswd} ) ) {
		$config{pktpswd} = '';
	}

	if ( defined( $config{sendnetmailtotg} ) ) {
		$config{sendnetmailtotg} = lc( $config{sendnetmailtotg} );
		if ( $config{sendnetmailtotg} ne 'yes' && 
									$config{sendnetmailtotg} ne 'no' ) {
			$config{sendnetmailtotg} = 'yes' ;
		}

	} else {
		$config{sendnetmailtotg} = 'no';
	}
#	$config{logfile} = catfile( $curpath, 'warnme.log' ) unless defined $config{logfile};

    return 1;
}

sub writelog($)
{
    my ( $str ) = @_;
    my ($sec,$min,$hour,$mday,$month,$year) = (localtime)[0...5];
    my $timestamp = sprintf("%04d-%02d-%02d %02d:%02d:%02d ",
                            $year+1900, $month+1, $mday, $hour, $min, $sec);
    $str =~ s/\n/\n$timestamp/g;
    if ( defined( $logfile ) ) {
        if ( open( my $FLOG, '>>', $logfile ) ) {
            print( $FLOG "$timestamp$str\n" );
            close( $FLOG );
        } else {
            print( STDERR "Can't open logfile $logfile. ($!)\n" );
        }
    }
    print( "$timestamp$str\n" );
}

sub tg_api_sendMessage($)
{
	my ( $Text ) = @_;
	my ( $error_message, $rc, $api );

	if ( !defined( $config{bot_token} ) || !defined( $config{api_url} ) ||
		!defined( $config{tg_uid} ) ) {
		writelog( 'Telegram messages disabled.' );
		return;
	}

	$api = WWW::Telegram::BotAPI->new(
    	token => $config{bot_token},
    	api_url => $config{api_url}
	);
    $rc = eval { $api->sendMessage (
	   {
		chat_id  => $config{tg_uid},
		text     => $Text,
#		entities => @t_entities,
#		disable_notification => $notifications,
#		allow_sending_without_reply => 1,
#		reply_to_message_id => $tg_msg_repl_to
	   }
    	)
    } or $error_message = $api->parse_error->{msg};
	if ( defined $error_message ) {
		writelog( $error_message );
	} else {
		writelog( 'Message to telegram sent.' );
	}
}

sub findndl($)
{
    my ( $filemask ) = @_;
    my $start = Time::HiRes::time();
    
    writelog("Finding last nodelist file from $filemask.");
    $filemask =~ /(.*?)([^\\\/]+)$/;
    my ( $ndlpath, $ndlfn ) = ( $1, $2 );
    my ( $nldate, $lastdate, $lastnl );
    $ndlpath = './' if !defined( $ndlpath ) || $ndlpath eq '';
    unless( -e $ndlpath ){
	print STDERR "\'$ndlpath\' does not exist!\n";
	exit;
    }
    $ndlfn =~ s/\./\\\./g;
    $ndlfn =~ s/\*/\.\*/g;
    $ndlfn =~ s/\?/\./g;
    $lastdate = 0;
    if ( opendir( DH, $ndlpath ) ) {
	while( readdir(DH) ) {
	    if( $_ =~ /^${ndlfn}$/i) {
		$nldate = read_ndl_time( catfile( $ndlpath, $_ ) );
		if ( $nldate > $lastdate ) {
		    $lastdate = $nldate;
		    $lastnl = $_;
		}
	    }
        }
    } else {
	print STDERR "Can't open $ndlpath. ($!)\n";
	exit;
    }
    unless( defined( $lastnl ) ) {
	print STDERR "Nodelist not found.\n";
	writelog( 'Nodelist not found.' );
	exit;
    }
    writelog( "Last nodelist \'$lastnl\' found at " . sprintf( "%.3f" , ( Time::HiRes::time() - $start ) ) . ' seconds.' );
    return catfile( $ndlpath, $lastnl );
}

sub read_ndl_time($)
{
   my ( $filename ) = @_;
   my ( $FH, $ndlstr, $line, $nltime, %month );

   $month{January} = 0;
   $month{February} = 1;
   $month{March} = 2;
   $month{April} = 3;
   $month{May} = 4;
   $month{June} = 5;
   $month{July} = 6;
   $month{August} = 7;
   $month{September} = 8;
   $month{October} = 9;
   $month{November} = 10;
   $month{December} = 11;
   
   if ( open( $FH, '<', $filename ) ) {
	$line = readline( $FH ); 
	close( $FH );
	if ( $line =~ /\;A \S+ Nodelist for \S+, (\S+) (\d+), (\d+) -- Day number \d+ : \d+/i ) {
		$nltime = timelocal( 0, 0, 0, $2, $month{$1}, $3 - 1900 );
		return $nltime;
	} else {
		print STDERR "Probably not a nodelist file $filename.\n";
		return 0;
	}
   } else {
	print STDERR "Cant read $filename ($!).";
	return 0;
   }
}

sub mktree($)
{
	my ( $msgbasefile ) = @_;
	my ( $err );
		make_path( $msgbasefile, { mode => 0777, error => \$err } ) unless -d $msgbasefile;
		if ( $err && @$err ) {
			for my $diag ( @$err ){
				my ($file, $message) = %$diag;
				if ($file eq '') {
					writelog( "General error: $message");
				} else {
            		writelog( "Problem making $file: $message");
        		}
    		}
			return 0 ;
		}
	return 1;
}


sub getbsoname($)
{
	my ( $address ) = @_;
	my ( $bsoname );

	$address =~ /(\d+)\:(\d+)\/(\d+)\.?(\d*)/;
	my ( $destzone, $destnet, $destnode, $destpnt ) = ( $1, $2, $3, $4 );
	$destpnt = 0 if !defined( $destpnt ) || $destpnt eq '';
	if ( $destzone == $config{defaultzone} ) {
#	    $bsoname = catdir( $config{outbound}, $config{defaultdomain} );
	    $bsoname = $config{outbound};
	    unless( -e $bsoname ) {
			print STDERR "Can't create $bsoname.\r" unless mktree( $bsoname );
	    }
#	} elsif ( defined($zonedom{$destzone}) ) {
#	    $bsoname = catfile( $outbound, $zonedom{$destzone} );
#	    unless( -e $bsoname ) {
#		print STDERR "Can't create $bsoname ($!).\r" unless make_path( $bsoname );
#	    }
	} else {
	    $bsoname = sprintf( "$config{outbound}\.%03x", $destzone );
	    unless( -e $bsoname ) {
			print STDERR "Can't create $bsoname.\r" unless mktree( $bsoname );
	    }
	}
	$bsoname = catfile( $bsoname, sprintf("%04x%04x", $destnet, $destnode) );
	if ( defined ( $destpnt ) && $destpnt != 0 ) {
	    $bsoname .= '.pnt';
	    unless( -e $bsoname ) {
			print STDERR "Can't create $bsoname.\r" unless mktree( $bsoname );
	    }
	    $bsoname = catfile( $bsoname, sprintf( "%08x", $destpnt ) );
	}

	return $bsoname;
}


sub packedmsg($$$$$$$;$$)
{
    my ( $msg_fromname, $msg_toname, $msg_fromaddr, $msg_toaddr, $attr,
         $msg_txt, $msg_subj, $msg_area, $msg_chrs ) = @_;

    my ( $msgheader,$m_txt,$msg_destzone,$msg_destnet,$msg_destnode,
         $msg_destpnt);

    if (length($msg_fromname) > 35){
	$msg_fromname = substr($msg_fromname,0,35);
    }
    $msg_fromname .= "\000";
    if (length($msg_toname) > 35){
	$msg_toname = substr($msg_toname,0,35);
    }
    $msg_toname .= "\000";
    if (length($msg_subj) > 71){
	$msg_subj = substr($msg_subj,0,71);
    }
    $msg_subj .= "\000";

    localtime =~ /[a-z]+ ([a-z]+)[ ]+(\d+) (\d+)\:(\d+)\:(\d+) \d\d(\d\d)/i;
    my $DateTime = sprintf("%02s", $2)." $1 $6  $3:$4:$5\000";

    $msg_fromaddr =~ /(\d+)\:(\d+)\/(\d+)\.?(\d*)/;
    
    my ( $msg_origzone, $msg_orignet, $msg_orignode, $msg_origpnt ) = ( $1, $2, $3, $4 );
    $msg_origpnt = 0 if !defined( $msg_origpnt ) || $msg_origpnt eq '';

    if ( defined($msg_toaddr) ) {
       $msg_toaddr =~ /(\d+)\:(\d+)\/(\d+)\.?(\d*)/;
       ($msg_destzone,$msg_destnet,$msg_destnode,$msg_destpnt) = ( $1, $2, $3, $4 );
       $msg_destpnt = 0 if !defined($msg_destpnt) || $msg_destpnt eq '';
    } else {
       ($msg_destzone,$msg_destnet,$msg_destnode,$msg_destpnt) = ( 0, 0, 0, 0 );
    }

    $msgheader = pack( "S7Z20", 2,$msg_orignode,$msg_destnode,
    $msg_orignet,$msg_destnet,$attr,0,$DateTime ) . $msg_toname .
    $msg_fromname . $msg_subj;

    $m_txt = "AREA:$msg_area\r" if defined($msg_area);
    my $TZUTC = sprintf( "%04d", tzoffset( localtime() ) );
    $m_txt .= "\001MSGID: $msg_fromaddr ".sprintf("%08x", time())."\r\001TZUTC: $TZUTC\r";
    
# NO INTL in Echmail permitted
    $m_txt .= "\001INTL $msg_destzone:$msg_destnet/$msg_destnode $msg_origzone:$msg_orignet/$msg_orignode\r" if !defined $msg_area;
    $m_txt .= "\001CHRS: ".uc($msg_chrs)." 2\r" if defined $msg_chrs;
    $m_txt .= "\001FMPT $msg_origpnt\r" if $msg_origpnt != 0;
    $m_txt .= "\001TOPT $msg_destpnt\r" if $msg_destpnt != 0 && !defined $msg_area;
    $msg_fromaddr =~ /(\d+\:\d+\/\d+\.?\d*)/;
    $msg_txt .= "\r--- \r \* Origin: perl packPKT by Stas Mishchenkov [2:460/58] \($1\)\r";

    my ( $g_sec, $g_min, $g_hour, $g_mday,$g_month,$g_year) = (gmtime)[0...5];
    if ( defined( $msg_area ) ) {
       if ( $msg_origpnt == 0 ) {
            $msg_txt .= "SEEN\-BY: $msg_orignet\/$msg_orignode\r".
	    "\x01PATH: $msg_orignet\/$msg_orignode\r";
       }
    } else {
	$msg_txt .= sprintf("\x01Via $msg_fromaddr \@%04d%02d%02d\.%02d%02d%02d\.UTC perl packPKT by Stas Mishchenkov\r",
	$g_year+1900, $g_month+1, $g_mday, $g_hour, $g_min, $g_sec );
    }
return "$msgheader$m_txt$msg_txt\000";
}

sub txt2pkt( $$$$;$$$$$$$$ )
{
    my $TZUTC = sprintf( "%04d", tzoffset( localtime() ) );
    my ($pkt_from,$pkt_to,$password,$msg_fromname, $msg_toname, $msg_fromaddr,
        $msg_toaddr, $attr, $msg_txt, $msg_subj, $msg_area, $msg_chrs ) = @_;
    my ($msg_destzone,$msg_destnet,$msg_destnode,$msg_destpnt);
    
    if (length($msg_fromname) > 35){
	$msg_fromname = substr($msg_fromname,0,35);
    }
    $msg_fromname .= "\000";
    if (length($msg_toname) > 35){
	$msg_toname = substr($msg_toname,0,35);
    }
    $msg_toname .= "\000";
    if (length($msg_subj) > 71){
	$msg_subj = substr($msg_subj,0,71);
    }
    $msg_subj .= "\000";

    $pkt_from =~ /(\d+)\:(\d+)\/(\d+)\.?(\d*)/;
    my ( $pkt_origzone, $pkt_orignet, $pkt_orignode, $pkt_origpnt ) = ( $1, $2, $3, $4 );
    $pkt_origpnt = 0 if !defined $pkt_origpnt || $pkt_origpnt eq '';
    
    $pkt_to =~ /(\d+)\:(\d+)\/(\d+)\.?(\d*)/;
    my ( $pkt_destzone,$pkt_destnet,$pkt_destnode,$pkt_destpnt ) = ( $1,$2,$3,$4 );
    $pkt_destpnt = 0 if !defined $pkt_destpnt || $pkt_destpnt eq '';
    $password = '' unless defined $password;
    $password .= "\x00" while length($password) < 8;
    localtime =~ /[a-z]+ ([a-z]+)[ ]+(\d+) (\d+)\:(\d+)\:(\d+) \d\d(\d\d)/i;
    my $DateTime = sprintf("%02s", $2)." $1 $6  $3:$4:$5\000";

    my ($second,$minute,$hour,$day,$month,$year,$wday,$yday,$isdst) = localtime();
    $year = $year + 1900;
    $yday++;

    my ($pkthdr,$msgheader,$m_txt);

    $msg_fromaddr =~ /(\d+)\:(\d+)\/(\d+)\.?(\d*)/;
    
    my ( $msg_origzone, $msg_orignet, $msg_orignode, $msg_origpnt ) = ( $1, $2, $3, $4 );
    $msg_origpnt = 0 if !defined( $msg_origpnt ) || $msg_origpnt eq '';

    if (defined($msg_toaddr)) {
       $msg_toaddr =~ /(\d+)\:(\d+)\/(\d+)\.?(\d*)/;
       ($msg_destzone,$msg_destnet,$msg_destnode,$msg_destpnt) = ( $1, $2, $3, $4 );
       $msg_destpnt = 0 if !defined($msg_destpnt) || $msg_destpnt eq '';
    } else {
       ($msg_destzone,$msg_destnet,$msg_destnode,$msg_destpnt) = ( 0, 0, 0, 0 );
    }

    $pkthdr = pack("S2S3S3 S2S2 C2A8S2S2 C2SS4I", $pkt_orignode,
    $pkt_destnode,$year,$month,
    $day,$hour,$minute,$second,0,2,$pkt_orignet,$pkt_destnet,3,3,
    $password,$pkt_origzone,$pkt_destzone,0, 0x0100,3,3,0x0001,
    $pkt_origzone,$pkt_destzone,$pkt_origpnt,$pkt_destpnt, 0 );

    $msgheader = pack( "S7Z20", 2,$msg_orignode,$msg_destnode,
    $msg_orignet,$msg_destnet,$attr,0,$DateTime ) . $msg_toname .
    $msg_fromname . $msg_subj;

    undef $msg_area if $msg_area eq '';
    $m_txt = "AREA:$msg_area\r" if defined($msg_area);
    $m_txt .= "\001MSGID: $msg_fromaddr ".sprintf("%08x", time())."\r\001TZUTC: $TZUTC\r";
    
# NO INTL in Echmail permitted
    $m_txt .= "\001INTL $msg_destzone:$msg_destnet/$msg_destnode $msg_origzone:$msg_orignet/$msg_orignode\r" if !defined $msg_area;
    $m_txt .= "\001CHRS: ". uc($msg_chrs)." 2\r" if defined $msg_chrs;
    $m_txt .= "\001FMPT $msg_origpnt\r" if $msg_origpnt != 0;
    $m_txt .= "\001TOPT $msg_destpnt\r" if $msg_destpnt != 0 && !defined $msg_area;
    $msg_fromaddr =~ /(\d+\:\d+\/\d+\.?\d*)/;
    $msg_txt .= "\r--- \r \* Origin: perl packPKT by Stas Mishchenkov [2:460/58] \($1\)\r";

    my ( $g_sec, $g_min, $g_hour, $g_mday,$g_month,$g_year) = (gmtime)[0...5];
    if ( defined( $msg_area ) ) {
       if ( $pkt_origpnt == 0 ) {
            $msg_txt .= "SEEN\-BY: $msg_orignet\/$msg_orignode $pkt_destnet\/$pkt_destnode\r".
	    "\x01PATH: $msg_orignet\/$msg_orignode\r";
       }
    } else {
	$msg_txt .= sprintf("\x01Via $pkt_from \@%04d%02d%02d\.%02d%02d%02d\.UTC perl packPKT by Stas Mishchenkov\r",
	$g_year+1900, $g_month+1, $g_mday, $g_hour, $g_min, $g_sec );
    }
    $msg_txt .= "\000\000\000";

return "$pkthdr$msgheader$m_txt$msg_txt";
}

sub writefile($$)
{
	my ( $filename, $buff ) = @_;
	my ( $HANDLE );

	my $tmp_fn = $filename . time();

	if ( open ( $HANDLE, '>', $tmp_fn ) ) {
	    binmode($HANDLE);
	    print( $HANDLE $buff );
	    close( $HANDLE );
	    print STDERR "Can't create $filename ($!)." unless move( $tmp_fn, $filename );
#	    chmod 0755, $filename if $^O eq 'linux';
#	    print "$filename saved.\n\n";
	} else {
	    print STDERR "Can't open $tmp_fn ($!).\n";
	    writelog( "Can't open $tmp_fn ($!)." );
	}
}

sub send_msg_as_bso ($)
{
	my ( $text ) = @_;

	my ( $bso, $fh, $pfh, $pkt, $msg, $size );

	if ( !defined( $config{fromaddress} ) || !defined( $config{toaddress} ) ||
		 !defined( $config{outbound} ) ) {
			writelog( 'Netmail messages disabled.' );
			return;
	}

	$bso = getbsoname( $config{toaddress} );
	sleep 1 while -e "$bso\.bsy";
	open( $fh, '>', "$bso\.bsy" );
	close( $fh );

	if ( -e "$bso\.dut" ) {
		$msg = packedmsg('Warning Robot', 'SysOp',$config{fromaddress},
						$config{toaddress}, 0x0181, $text, 'Warning!',
						undef, 'CP866' );

		$size = (stat("$bso\.dut"))[7];

		if ( open( $pfh, '+<', "$bso\.dut" ) ) {
			binmode( $pfh );
			seek( $pfh, ($size - 2), 0 );
			print $pfh "$msg\000\000";
			close( $pfh );
		} else {
			writelog( "Can't open $bso\.dut. ($!)" );
		}
	} else {
		$pkt = txt2pkt( $config{fromaddress}, $config{toaddress},
					$config{pktpswd}, 'Warning Robot', 'SysOp',
					$config{fromaddress}, $config{toaddress}, 0x0181, $text,
					'Warning!', '',	'CP866' );
		writefile( "$bso\.dut", $pkt );
	}
	unlink "$bso\.bsy";
}


sub readlastread
{
    my $lrfile = catfile($_[0], "lastread");
    my ( $LRF );
    return (1,0) unless -e $lrfile;
    return (1,0) if -d $lrfile;
    my $lrsize = -s $lrfile;
    my $usercount = $lrsize/2;
    if ( open( $LRF, '<', $lrfile) )
    {
        binmode($LRF);
        read( $LRF, my $lr, $lrsize );
        close($LRF);
        return unpack("S$usercount", $lr);

    } else {
        writelog("Can't open $lrfile. $!");
        return (1,0);
    }
}

sub writelastread
{
    my ($lastreadDir, @lastreads) = @_;

	my $users = @lastreads;

    my ( $FLTD, $lastreadPath, $i );

	for ( $i = 0; $i < $users; $i++ ) {
		$lastreads[$i] = 0 if !defined $lastreads[$i];
	}

    $lastreadPath = catfile($lastreadDir, "lastread");

    if( open( $FLTD, '>', $lastreadPath ) )
    {
        binmode($FLTD);
        print( $FLTD pack( "S$users", @lastreads ) );
        close( $FLTD );
    }
    else
    {
        writelog("Can't open \"$lastreadPath\".($!)");
    }
}

sub newmail()
{
    my @lastread = readlastread( $config{netmail} );
	my ( $proguserno, $maxlr, $unread ) = ( 255, 0, 0 );
	my $users = @lastread;

	if ( $users < $proguserno ) {
		$users = $proguserno;
		$lastread[$proguserno] = 0;
	}

	if ( $lastread[$config{fidouserno}] > $lastread[$proguserno] ) {
		$maxlr = $lastread[$config{fidouserno}];
	} else {
		$maxlr = $lastread[$proguserno];
	}

	if ( opendir( my $DH, $config{netmail} ) ) {
		while( readdir( $DH ) ) {
			next unless $_ =~ /^(\d+)\.msg$/i;
			if ($1 > $maxlr) {
				$unread++;
				$lastread[$proguserno] = $1 if $lastread[$proguserno] < $1;
				push( @tgmsgs,
					 perp2tg( readmsg( catfile( $config{netmail}, $_ ) ) ) );
			}
		}
		closedir($DH);
	} else {
		writelog( "Can't open $config{netmail}. ($!)" );
	}
	writelastread( $config{netmail}, @lastread );

	return $unread;
}


sub readmsg
{
    my ($msgpath) = @_;
    my $msgsize = -s $msgpath;
    if ( open(FM, "<$msgpath") )
    {
        binmode(FM);
        read(FM, my $msg, $msgsize);
        close(FM);
        return ( unpack("Z36Z36Z72Z20S13Z".($msgsize - 190), $msg) );
    } else {
        print(STDERR "Can't open file \"$msgpath\". \($!\)\n");
        writelog("Can't open \"$msgpath\".($!)");
        return 0;
    }
}

sub perp2tg
{
	my ($fromUserName,$toUserName,$subject,$DateTime,
    $timesread,$destnode,$orignode,$cost,$orignet,$destnet,
    $date_written16,$time_written16,$date_arrived,$time_arrived,$replyto,
    $attr,$nextmsg, $msgtext) = @_;

	my ( $tgmsgtxt, $fromaddress, $fromchrs, $cutlen, @msgs );

	if ( $msgtext =~ /\x01CHRS\:? ([^\r\n]+) \d+\r/ ) {
		$fromchrs = lc($1);
	} else {
		$fromchrs = 'cp866';
	}

	if ( $msgtext =~ /\x0d \* Origin: [^\r\n]*\((\d+\:\d+\/\d+.?\d*)\)\x0d\x00$/ ) {
		$fromaddress = $1;
	} elsif ( $msgtext =~ /\x01MSGID: (\d+\:\d+\/\d+.?\d*) [^\r\n]+\r/ ) {
		$fromaddress = $1;
	} elsif ( $msgtext =~ /\x01INTL\:? \d+\:\d+\/\d+ (\d+\:\d+\/\d+)/ ) {
		$fromaddress = $1;
		$fromaddress = "$fromaddress\.$1" if $msgtext =~ /\x01FMPT\:? (\d+)\r/;
	} else {
		$fromaddress = "$config{defaultzone}\:$orignode\/$orignet";
		$fromaddress = "$fromaddress\.$1" if $msgtext =~ /\x01FMPT\:? (\d+)\r/;
	}

	$msgtext =~ s/\x01[^\r\n]*\r//g;
	$msgtext =~ s/\-\-\- [^\r\n]*\r//;
	$msgtext =~ s/ \* Origin: [^\r\n]*\r//;
	$msgtext =~ s/[\r\n]begin \d+[ ]+([^\r\n]+)[\r\n][^ ]*?[\r\n]end[\r\n]/\nUUe of file $1 skipped.\n/gi;
	$msgtext =~ s/\r/\n/g;

	$tgmsgtxt = "From: $fromUserName $fromaddress\n".
				"Subj: $subject\n\n$msgtext";

	my $maxlen = 3850;
	my $ofset = 0;
	my $filelen = length($tgmsgtxt);
	while( $ofset < $filelen ) {
	    $cutlen = index( $tgmsgtxt, ' ', $ofset + $maxlen - 80 );
	    $cutlen = $maxlen if ( $cutlen == -1 || 
								$cutlen == 0 || !defined($cutlen) );
	    $cutlen++;
	    $cutlen = $maxlen if $cutlen > $maxlen;

		push( @msgs, decode( 
							$fromchrs, substr( $tgmsgtxt, $ofset, $cutlen )."\n" 
							) 
			);
	    $ofset += $cutlen;
	}
	
#	$tgmsgtxt = decode( $fromchrs, $tgmsgtxt );

	return @msgs;
}

# -- MAIN ---->

  my $oneday = 86400;

  my $options = join( ' ', @ARGV );

  GetOptions (
            "config=s"    => \$configfile,
            "help"        => \$needhelp,
            "nodelist"    => \$warnnodelist,
			"mail"        => \$warnnetmail,
            "log=s"       => \$logfile
	    )or die("Error in command line arguments\n");

  if ( defined( $needhelp ) || !defined( $configfile ) ) {
    usage();
  }

  readconf();

  $logfile = $config{logfile} unless defined $logfile;

  writelog( "[$$] $programfile (v.$vers) $options" );

  if ( $warnnodelist ) {
      $daysold = int( ( time() - read_ndl_time( findndl( $config{nodelist} ) ) ) / $oneday );
	  if ( $daysold > $config{days} ) {
		writelog( "Nodelist is $daysold old." );
		tg_api_sendMessage("Nodelist is  $daysold old.");
		send_msg_as_bso ( "\rNodelist is  $daysold old.\r" );
	  } else {
		writelog( "Nodelist is actual." );
	  }
  }

  if ( $warnnetmail ) {
		$newmailcount = newmail();
		if ( $newmailcount > 0 ) {
			writelog( "You have $newmailcount new netmail messages." );
			tg_api_sendMessage("You have $newmailcount new netmail messages.");
			if ( $config{sendnetmailtotg} eq 'yes' ) {
				foreach my $msg ( @tgmsgs ) {
					tg_api_sendMessage( $msg );
				}
			}
		}
  }
